//	TorusGames2DCrossword.c
//
//	The crossword puzzle's cells are indexed by coordinates (h,v), with
//	h running left-to-right from 0 to n-1 and
//	v running bottom-to-top from 0 to n-1,
//	where n is of course the size of the puzzle.
//	I chose to have v run from bottom to top
//	to agree with Metal's bottom-up Normalized Device Coordinates.
//	Cell centers align with the natural n×n grid, so for example
//	cells straddle the edges of the fundamental domain
//	and a single cell sits centered at the fundamental domain's corners.
//	In other words, cell (h,v) sits with its center at
//	( -1/2 + h/n,  -1/2 + v/n ).
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt


#include "TorusGames-Common.h"
#include "GeometryGamesLocalization.h"
#include "GeometryGamesSound.h"
#include <math.h>


//	The following option is useful while creating crossword puzzles,
//	in particular for creating a screenshot to show the translator
//	what a given puzzle type looks like.
//#define SHOW_SOLUTION

#define NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_CROSSWORD		1	//	texture is plain white


typedef enum
{
	WordHorizontal,		//	→ or ←
	WordVertical		//	always ↓
} WordOrientation;


//	Public functions with private names
static void					CrosswordReset(ModelData *md);
static void					CrosswordHandMoved(ModelData *md);
static bool					CrosswordDragBegin(ModelData *md, bool aRightClick);
static TitledErrorMessage	*CrosswordCharacterInput(ModelData *md, Char16 aCharacter);
static void					CrosswordSimulationUpdate(ModelData *md);
static void					CrosswordRefreshMessage(ModelData *md);

//	Private functions
static void					SetHotWordFromCursor(ModelData *md);
static void					SetHotWord(ModelData *md, unsigned int h, unsigned int v, bool aFlip, WordOrientation aWordOrientation);
static bool					MoveToPrecedingNonblackCell(ModelData *md, unsigned int *h, unsigned int *v, bool *aFlip, Char16 aDirection, bool aCyclicFlag);
static bool					MoveToNextNonblackCell(ModelData *md, unsigned int *h, unsigned int *v, bool *aFlip, Char16 aDirection, bool aCyclicFlag);
static bool					RespondToArrow(ModelData *md, Char16 aCharacter);
static bool					CrosswordIsSolved(ModelData *md);
static void					FixAccentsAsNecessary(ModelData *md);


void Crossword2DSetUp(ModelData *md)
{
	const Char16	*theLanguageCode;
	unsigned int	h,
					v;

	//	Initialize function pointers.
	md->itsGameShutDown					= NULL;
	md->itsGameReset					= &CrosswordReset;
	md->itsGameHumanVsComputerChanged	= NULL;
	md->itsGame2DHandMoved				= &CrosswordHandMoved;
	md->itsGame2DDragBegin				= &CrosswordDragBegin;
	md->itsGame2DDragObject				= NULL;
	md->itsGame2DDragEnd				= NULL;
	md->itsGame3DDragBegin				= NULL;
	md->itsGame3DDragObject				= NULL;
	md->itsGame3DDragEnd				= NULL;
	md->itsGame3DGridSize				= NULL;
	md->itsGameCharacterInput			= &CrosswordCharacterInput;
	md->itsGameSimulationUpdate			= &CrosswordSimulationUpdate;
	md->itsGameRefreshMessage			= &CrosswordRefreshMessage;

	//	Initialize variables for robust error handling.
	md->itsGameOf.Crossword2D.itsPuzzleSize	= 0;
	
	//	Note the initial language.
	theLanguageCode = GetCurrentLanguage();
	md->itsGameOf.Crossword2D.itsPreviousLanguageCode[0] = theLanguageCode[0];
	md->itsGameOf.Crossword2D.itsPreviousLanguageCode[1] = theLanguageCode[1];
	md->itsGameOf.Crossword2D.itsPreviousLanguageCode[2] = 0;

	//	Start at the beginning of each puzzle list.
	md->itsGameOf.Crossword2D.itsIndexTorus = 0;
	md->itsGameOf.Crossword2D.itsIndexKlein = 0;

	//	Initialize the hot cell, just for good form.
	//	CrosswordReset() will reset it to avoid the black cells.
	md->itsGameOf.Crossword2D.itsHotCellH		= 0;
	md->itsGameOf.Crossword2D.itsHotCellV		= 0;
	md->itsGameOf.Crossword2D.itsHotDirection	= u'→';

	//	Set up an empty board to ensure safe texture initialization.
	for (h = 0; h < MAX_CROSSWORD_SIZE; h++)
		for (v = 0; v < MAX_CROSSWORD_SIZE; v++)
			md->itsGameOf.Crossword2D.itsBoard[h][v] = ' ';
	
	//	Reset the crossword puzzle.
	CrosswordReset(md);
}


const Char16 *ChooseCrosswordCellFont(void)
{
	//	Choose a font that looks good and is available on the host platform.
#if TARGET_OS_OSX
	if (IsCurrentLanguage(u"ja"))
	{
		//	"Hiragino Kaku Gothic Pro W3" and "Hiragino Kaku Gothic Pro W6"
		//		are gothic fonts of lighter and heavier line weights.
		//	"Hiragino Mincho Pro W3" and "Hiragino Mincho Pro W6"
		//		are mincho fonts of lighter and heavier line weights.
		//	"Arial Unicode MS" is a gothic font with a fairly heavy line weight.
		//
		//	Tatsu says
		//
		//		Kaku-Gothic W3 seems best.
		//
		return u"Hiragino Kaku Gothic Pro W3";
	}
	else
	if (IsCurrentLanguage(u"ko"))
	{
		//	"AppleMyungjo Regular" is easy to read,
		//		yet still has a human touch.
		//	"AppleGothic Regular" is very easy to read,
		//		but lacks AppleMyungjo's human touch.
		//	"GungSeo Regular" (궁서) has personality,
		//		but the characters seems a little small,
		//		while the spacing between them seems a little large.
		//	"PilGi Regular" looks like handwriting.
		//		OK, but maybe a little hard to read.
		//	"Arial Unicode MS" is like "AppleGothic Regular"
		//		but slightly bolder.
		//
		//	Note that macOS is fussy about font names:
		//	"GungSeo Regular" works while "GungSeo" fails, yet
		//	"Arial Unicode MS Regular" fails while "Arial Unicode MS" works.
		//	Go figure.
		//
		//	Among those fonts, 현종 prefers AppleGothic.
		//
		return u"AppleGothic Regular";
	}
	else
	if (IsCurrentLanguage(u"zs"))
	{
		//	For options, the page
		//		http://www.yale.edu/chinesemac/pages/fonts.html
		//	contains a chart "Fonts via Apple" that says
		//	which fonts are included in which versions of OS.
		return u"Heiti SC";
	}
	else
	if (IsCurrentLanguage(u"zt"))
	{
		//	For options, see comment immediately above.
		return u"Heiti TC";
	}
	else
	{
		return u"Helvetica";
	}
#elif TARGET_OS_IOS
	//	The page
	//
	//		http://iosfonts.com
	//
	//	shows which fonts are included in which versions of iOS.
	//
	if (IsCurrentLanguage(u"ja"))
	{
		//	"HiraKakuProN-W3" and "HiraKakuProN-W6"
		//		are gothic fonts of lighter and heavier line weights.
		//	"HiraMinProN-W3" and "HiraMinProN-W6"
		//		are mincho fonts of lighter and heavier line weights,
		//		but are available on iPad only, not on iPhone.
		//
		//	Tatsu says
		//
		//		Kaku-Gothic W3 seems best.
		//
		return u"HiraKakuProN-W3";
	}
	else
	if (IsCurrentLanguage(u"ko"))
	{
		//	"AppleGothic" is the only font from the above-mentioned lists
		//		that I recognize as supporting Korean.
		//
		//	Conveniently, this is the font that 현종 prefers.
		//
		return u"AppleGothic";
	}
	else
	if (IsCurrentLanguage(u"zs"))
	{
		//	Reasonable choices are "STHeitiSC-Light" or ""STHeitiSC-Medium".
		return u"STHeitiSC-Light";
	}
	else
	if (IsCurrentLanguage(u"zt"))
	{
		//	For options, see comment immediately above.
		//	For consistency, use the "TC" variant of the "SC" font chosen above.
		return u"STHeitiTC-Light";
	}
	else
	{
		return u"Helvetica";
	}
#else
#error Crossword cell font not specified
#endif
}


static void CrosswordReset(ModelData *md)
{
	ErrorText		theErrorMessage		= NULL;
	Byte			theClueCount;
	unsigned int	theCharacterCount,
					h,
					v;
	const Char16	*theLanguageCode;
	unsigned int	theNumPuzzles,
					*theIndex;
	Char16			theRawPuzzleData[2048],
					*r,				//	the read position
					theDirection;	//	the arrow direction:  ← → ↓

	//	Initialize the clues to empty values.
	for (theClueCount = 0; theClueCount < MAX_CROSSWORD_CLUES; theClueCount++)
		for (theCharacterCount = 0; theCharacterCount < MAX_CROSSWORD_CLUE_LENGTH; theCharacterCount++)
			md->itsGameOf.Crossword2D.itsCluesText[theClueCount][theCharacterCount] = 0;
	for (h = 0; h < MAX_CROSSWORD_SIZE; h++)
		for (v = 0; v < MAX_CROSSWORD_SIZE; v++)
		{
			md->itsGameOf.Crossword2D.itsClueIndexAcross[h][v] = CROSSWORD_NO_CLUE;
			md->itsGameOf.Crossword2D.itsClueIndexDown  [h][v] = CROSSWORD_NO_CLUE;
			md->itsGameOf.Crossword2D.itsLeftwardWord   [h][v] = false;
		}
	
	//	If the user changed languages, reset the puzzle indices to 0 
	//	so the user will get the easiest puzzle in the new language.
	if ( ! IsCurrentLanguage(md->itsGameOf.Crossword2D.itsPreviousLanguageCode) )
	{
		md->itsGameOf.Crossword2D.itsIndexTorus = 0;
		md->itsGameOf.Crossword2D.itsIndexKlein = 0;

		theLanguageCode = GetCurrentLanguage();
		md->itsGameOf.Crossword2D.itsPreviousLanguageCode[0] = theLanguageCode[0];
		md->itsGameOf.Crossword2D.itsPreviousLanguageCode[1] = theLanguageCode[1];
		md->itsGameOf.Crossword2D.itsPreviousLanguageCode[2] = 0;
	}

	//	Read the raw puzzle data as a string resource.
	theNumPuzzles	= GetNumPuzzles(Game2DCrossword, md->itsTopology);
	theIndex		= md->itsTopology == Topology2DTorus ?
						&md->itsGameOf.Crossword2D.itsIndexTorus :
						&md->itsGameOf.Crossword2D.itsIndexKlein;
	*theIndex		%= theNumPuzzles;	//	unnecessary but safe
#ifdef GAME_CONTENT_FOR_SCREENSHOT
	//	Get the desired puzzle for the screenshot.
	GEOMETRY_GAMES_ASSERT(md->itsTopology == Topology2DTorus, "Torus screenshots expected");
	if (IsCurrentLanguage(u"de")) *theIndex = 1;
	if (IsCurrentLanguage(u"el")) *theIndex = 0;
	if (IsCurrentLanguage(u"en")) *theIndex = 1;
	if (IsCurrentLanguage(u"es")) *theIndex = 1;
	if (IsCurrentLanguage(u"fi")) *theIndex = 0;
	if (IsCurrentLanguage(u"fr")) *theIndex = 4;
	if (IsCurrentLanguage(u"it")) *theIndex = 4;
	if (IsCurrentLanguage(u"ja")) *theIndex = 0;
	if (IsCurrentLanguage(u"ko")) *theIndex = 3;
	if (IsCurrentLanguage(u"nl")) *theIndex = 5;
	if (IsCurrentLanguage(u"pt")) *theIndex = 3;
	if (IsCurrentLanguage(u"ru")) *theIndex = 5;
	if (IsCurrentLanguage(u"vi")) *theIndex = 1;
	if (IsCurrentLanguage(u"zs")) *theIndex = 0;
	if (IsCurrentLanguage(u"zt")) *theIndex = 0;
#endif
	//	Get the puzzle with the current topology and index.
	GetRawPuzzleData(	Game2DCrossword,
						md->itsTopology,
						*theIndex,
						theRawPuzzleData,
						BUFFER_LENGTH(theRawPuzzleData));

	//	Increment the index for next time.
	(*theIndex)++;
	*theIndex %= theNumPuzzles;
	if (IsCurrentLanguage(u"zt"))
	{
		//	Two of the Chinese crossword puzzles work fine in simplified characters,
		//	but don't work in traditional characters, because a shared simplfied character
		//	gets converted to one traditional character in the horizontal word
		//	and to a different traditional character in the vertical word.
		//	In traditional Chinese, skip the puzzles that suffer this problem.
		if (md->itsTopology == Topology2DTorus)
		{
			//	CAUTION:  THESE PUZZLE NUMBERS ARE HARD-CODED!
			if (*theIndex == 1 || *theIndex == 5)
				*theIndex = (*theIndex + 1) % theNumPuzzles;
		}
		else	//	Topology2DKlein
		{
			//	Both Klein bottle puzzles are OK in traditional Chinese.
		}
	}

	//	Did the raw data arrive OK?
	if (theRawPuzzleData[0] == 0)
	{
		theErrorMessage = u"Couldn't load raw crossword puzzle data.";
		goto CleanUpCrosswordReset;
	}

	//	Initialize a "read pointer".
	r = theRawPuzzleData;

	//	Confirm the leading 'T' or 'K'.
	if (*r++ != (md->itsTopology == Topology2DTorus ? 'T' : 'K'))
	{
		theErrorMessage = u"Puzzle data lacks leading 'T' or 'K'.";
		goto CleanUpCrosswordReset;
	}

	//	Parse the puzzle size.
	if (r[0] >= '1' && r[0] <= '9'	//	1-digit puzzle size
	 && r[1] == '/' && r[2] == '/')
	{
		md->itsGameOf.Crossword2D.itsPuzzleSize = r[0] - '0';
		r += 3;	//	Skip past the two separators "//" as well as puzzle size itself.
	}
	else
	if (r[0] == '1'
	 && r[1] >= '0' && r[1] <= '9'	//	2-digit puzzle size
	 && r[2] == '/' && r[3] == '/')
	{
		md->itsGameOf.Crossword2D.itsPuzzleSize = 10*(r[0] - '0') + (r[1] - '0');
		r += 4;	//	Skip past the two separators "//" as well as puzzle size itself.
	}
	else
	{
		theErrorMessage = u"Puzzle data must begin with <T|K><puzzle size>//, for example T5// or K4// .  Make sure MAX_CROSSWORD_SIZE is at least as large as your puzzle size.";
		goto CleanUpCrosswordReset;
	}
	
	if (md->itsGameOf.Crossword2D.itsPuzzleSize > MAX_CROSSWORD_SIZE)
	{
		theErrorMessage = u"Puzzle size exceeds MAX_CROSSWORD_SIZE.";
		goto CleanUpCrosswordReset;
	}

	//	Read the puzzle solution.
	//
	//	Note #1:  The internal v-coordinate runs bottom-to-top,
	//	while raw puzzle data runs top-to-bottom.
	//
	//	Note #2:  The internal coordinates are ordered (h,v),
	//	so the outer loop iterates over v, even though v is
	//	the second coordinate.
	for (v = md->itsGameOf.Crossword2D.itsPuzzleSize; v-- > 0; )
	{
		for (h = 0; h < md->itsGameOf.Crossword2D.itsPuzzleSize; h++)
		{
			if (*r != 0		//	essential for robust string parsing
			 && *r != '/')	//	not essential, but convenient,
							//	  because '/' most likely serves only
							//		as a marker not as puzzle data
			{
				md->itsGameOf.Crossword2D.itsSolution[h][v] = *r++;
			}
			else
			{
				theErrorMessage = u"Incomplete puzzle solution data.";
				goto CleanUpCrosswordReset;
			}
		}

		if (*r++ != '/')
		{
			theErrorMessage = u"Missing '/' between puzzle solution lines (or some line is too long).";
			goto CleanUpCrosswordReset;
		}
	}
	if (*r++ != '/')
	{
		theErrorMessage = u"Missing '//' at end of puzzle solution.";
		goto CleanUpCrosswordReset;
	}

	//	Read the clues.
	theClueCount = 0;
	while (*r != '/')
	{
		if (theClueCount >= MAX_CROSSWORD_CLUES)
		{
			theErrorMessage = u"Too many clues.";
			goto CleanUpCrosswordReset;
		}

		if (*r == 0)
		{
			theErrorMessage = u"Missing // at end of last clue.";
			goto CleanUpCrosswordReset;
		}

		//	Read the clue's position and direction.

		h = *r++ - '0';
		if (h >= md->itsGameOf.Crossword2D.itsPuzzleSize)	//	h is unsigned so h >= 0 is guaranteed
		{
			theErrorMessage = u"Clue has bad h coordinate.";
			goto CleanUpCrosswordReset;
		}

		v = *r++ - '0';
		if (v >= md->itsGameOf.Crossword2D.itsPuzzleSize)	//	v is unsigned so v >= 0 is guaranteed
		{
			theErrorMessage = u"Clue has bad v coordinate.";
			goto CleanUpCrosswordReset;
		}

		theDirection = *r++;
		if (theDirection != u'←'
		 && theDirection != u'→'
		 && theDirection != u'↓')
		{
			theErrorMessage = u"Clue has no direction arrow.";
			goto CleanUpCrosswordReset;
		}

		//	Read the clue's text.
		theCharacterCount = 0;
		while (*r != '/')
		{
			if (*r == 0)
			{
				theErrorMessage = u"Incomplete clue.";
				goto CleanUpCrosswordReset;
			}

			if (theCharacterCount >= MAX_CROSSWORD_CLUE_LENGTH - 1)
			{
				theErrorMessage = u"Clue is too long.";
				goto CleanUpCrosswordReset;
			}

			md->itsGameOf.Crossword2D.itsCluesText[theClueCount][theCharacterCount] = *r++;

			theCharacterCount++;
		}
		md->itsGameOf.Crossword2D.itsCluesText[theClueCount][theCharacterCount] = 0;
		r++;

		//	Assign the clue to the appropriate puzzle squares.
		switch (theDirection)
		{
			case u'←':
				while (md->itsGameOf.Crossword2D.itsSolution[h][v] != '*')
				{
					if (md->itsGameOf.Crossword2D.itsClueIndexAcross[h][v] != CROSSWORD_NO_CLUE)
					{
						theErrorMessage = u"Puzzle contains either an unbounded (cyclic) horizontal word or overlapping horizontal clues.";
						goto CleanUpCrosswordReset;
					}

					md->itsGameOf.Crossword2D.itsClueIndexAcross[h][v] = theClueCount;
					md->itsGameOf.Crossword2D.itsLeftwardWord   [h][v] = true;
					if (h > 0)
						h--;
					else
						h = md->itsGameOf.Crossword2D.itsPuzzleSize - 1;
				}
				break;

			case u'→':
				while (md->itsGameOf.Crossword2D.itsSolution[h][v] != '*')
				{
					if (md->itsGameOf.Crossword2D.itsClueIndexAcross[h][v] != CROSSWORD_NO_CLUE)
					{
						theErrorMessage = u"Puzzle contains either an unbounded (cyclic) horizontal word or overlapping horizontal clues.";
						goto CleanUpCrosswordReset;
					}

					md->itsGameOf.Crossword2D.itsClueIndexAcross[h][v] = theClueCount;
					if (h < md->itsGameOf.Crossword2D.itsPuzzleSize - 1)
						h++;
					else
						h = 0;
				}
				break;

			case u'↓':
				while (md->itsGameOf.Crossword2D.itsSolution[h][v] != '*')
				{
					if (md->itsGameOf.Crossword2D.itsClueIndexDown[h][v] != CROSSWORD_NO_CLUE)
					{
						theErrorMessage = u"Puzzle contains either an unbounded (cyclic) vertical word or overlapping vertical clues.";
						goto CleanUpCrosswordReset;
					}

					md->itsGameOf.Crossword2D.itsClueIndexDown[h][v] = theClueCount;
					if (v > 0)
						v--;
					else
					{
						v = md->itsGameOf.Crossword2D.itsPuzzleSize - 1;
						if (md->itsTopology == Topology2DKlein)
							h = (h > 0) ? md->itsGameOf.Crossword2D.itsPuzzleSize - h : 0;
					}
				}
				break;
		}

		//	Increment the count.
		theClueCount++;
	}

	//	Initialize the current board to all ' '.
	for (h = 0; h < md->itsGameOf.Crossword2D.itsPuzzleSize; h++)
	{
		for (v = 0; v < md->itsGameOf.Crossword2D.itsPuzzleSize; v++)
		{
#ifdef SHOW_SOLUTION
			md->itsGameOf.Crossword2D.itsBoard[h][v]		= md->itsGameOf.Crossword2D.itsSolution[h][v];
#else
			md->itsGameOf.Crossword2D.itsBoard[h][v]		= u' ';
#endif
			md->itsGameOf.Crossword2D.itsBoardFlips[h][v]	= false;
		}
	}

#ifdef GAME_CONTENT_FOR_SCREENSHOT
//	Define a macro ESL to "expose solution letter".
#define ESL(h,v)	md->itsGameOf.Crossword2D.itsBoard[h][v] = md->itsGameOf.Crossword2D.itsSolution[h][v];
	//	Expose some words from the solution.
	if (IsCurrentLanguage(u"de")) { ESL(4,2) ESL(4,1) ESL(4,0) ESL(4,4) ESL(1,0) ESL(2,0) ESL(3,0) }
	if (IsCurrentLanguage(u"el")) { ESL(0,1) ESL(0,0) ESL(0,5) ESL(0,4) ESL(5,5) ESL(1,5) ESL(2,5) }
	if (IsCurrentLanguage(u"en")) { ESL(3,3) ESL(3,2) ESL(3,1) ESL(3,0) ESL(4,1) ESL(0,1) ESL(1,1) }
	if (IsCurrentLanguage(u"es")) { ESL(3,1) ESL(4,1) ESL(0,1) ESL(1,1) ESL(4,2) ESL(4,0) ESL(4,4) }
	if (IsCurrentLanguage(u"fi")) { ESL(4,0) ESL(4,5) ESL(4,4) ESL(4,3) ESL(4,2) ESL(3,5) ESL(5,5) ESL(0,5) ESL(1,5) }
	if (IsCurrentLanguage(u"fr")) { ESL(4,3) ESL(4,2) ESL(4,1) ESL(4,0) ESL(3,2) ESL(0,2) ESL(1,2) }
	if (IsCurrentLanguage(u"it")) { ESL(2,1) ESL(2,0) ESL(2,4) ESL(2,3) ESL(1,0) ESL(3,0) ESL(4,0) }
	if (IsCurrentLanguage(u"ja")) { ESL(1,2) ESL(1,1) ESL(1,0) ESL(2,2) ESL(3,2) }
	if (IsCurrentLanguage(u"ko")) { ESL(6,4) ESL(0,4) ESL(1,4) ESL(2,4) ESL(0,6) ESL(1,6) ESL(2,6) ESL(3,6) ESL(1,5) ESL(0,3) ESL(0,2) }
	if (IsCurrentLanguage(u"nl")) { ESL(3,1) ESL(3,0) ESL(3,4) ESL(3,3) ESL(1,0) ESL(2,0) ESL(4,0) }
	if (IsCurrentLanguage(u"pt")) { ESL(2,3) ESL(2,2) ESL(2,1) ESL(2,0) ESL(4,1) ESL(0,1) ESL(1,1) }
	if (IsCurrentLanguage(u"ru")) { ESL(4,2) ESL(4,1) ESL(4,0) ESL(4,4) ESL(3,1) ESL(0,1) ESL(1,1) }
	if (IsCurrentLanguage(u"vi")) { ESL(4,3) ESL(4,2) ESL(4,1) ESL(4,0) ESL(3,2) ESL(0,2) ESL(1,2) }
	if (IsCurrentLanguage(u"zs")) { ESL(4,3) ESL(4,2) ESL(4,1) ESL(4,0) ESL(3,1) ESL(5,1) ESL(0,1) }
	if (IsCurrentLanguage(u"zt")) { ESL(4,3) ESL(4,2) ESL(4,1) ESL(4,0) ESL(3,1) ESL(5,1) ESL(0,1) }
#endif

	//	Initialize the hot word to the first horizontal word in the top row.
	//	(Hmmm... a singleton word is a theoretical possibility.
	//	Fortunately I don't think this occurs in any of the puzzles,
	//	and even if it did it shouldn't be a disaster.)
	h = 0;
	v = md->itsGameOf.Crossword2D.itsPuzzleSize - 1;
	while (md->itsGameOf.Crossword2D.itsSolution[h][v] == '*')
	{
		h++;
		if (h == md->itsGameOf.Crossword2D.itsPuzzleSize)
		{
			theErrorMessage = u"Crossword puzzle's top row contains all black squares.";
			goto CleanUpCrosswordReset;
		}
	}

	//	Move the hot cell to the beginning of the hot word.
	//	In a Klein bottle puzzle, the word may begin from either the left or the right.
	while (MoveToPrecedingNonblackCell(	md,
										&h,
										&v,
										NULL,
										md->itsGameOf.Crossword2D.itsLeftwardWord[h][v] ? u'←' : u'→',
										false))
		;

	//	Color the hot word.
	//	(SetHotWord() will call CrosswordRefreshMessage() only if the hot cell has changed.)
	SetHotWord(md, h, v, false, WordHorizontal);

#ifdef GAME_CONTENT_FOR_SCREENSHOT
	if (IsCurrentLanguage(u"de")) SetHotWord(md, 4, 2, false, WordVertical	);
	if (IsCurrentLanguage(u"el")) SetHotWord(md, 0, 1, false, WordVertical	);
	if (IsCurrentLanguage(u"en")) SetHotWord(md, 3, 3, false, WordVertical	);
	if (IsCurrentLanguage(u"es")) SetHotWord(md, 3, 1, false, WordHorizontal);
	if (IsCurrentLanguage(u"fi")) SetHotWord(md, 4, 0, false, WordVertical	);
	if (IsCurrentLanguage(u"fr")) SetHotWord(md, 4, 3, false, WordVertical	);
	if (IsCurrentLanguage(u"it")) SetHotWord(md, 2, 1, false, WordVertical	);
	if (IsCurrentLanguage(u"ja")) SetHotWord(md, 1, 2, false, WordVertical	);
	if (IsCurrentLanguage(u"ko")) SetHotWord(md, 6, 4, false, WordHorizontal);
	if (IsCurrentLanguage(u"nl")) SetHotWord(md, 3, 1, false, WordVertical	);
	if (IsCurrentLanguage(u"pt")) SetHotWord(md, 2, 3, false, WordVertical	);
	if (IsCurrentLanguage(u"ru")) SetHotWord(md, 4, 2, false, WordVertical	);
	if (IsCurrentLanguage(u"vi")) SetHotWord(md, 4, 3, false, WordVertical	);
	if (IsCurrentLanguage(u"zs")) SetHotWord(md, 4, 3, false, WordVertical	);
	if (IsCurrentLanguage(u"zt")) SetHotWord(md, 4, 3, false, WordVertical	);
#endif

#ifdef CREATE_CROSSWORD_FOR_TALK
	//	Replace all black cells with empty white cells that can
	//	accept character input, for use when designing puzzles by hand.
	//
	//		Kludge alert:  It's possible that this could cause trouble
	//		elsewhere in the code, because words are no longer
	//		guaranteed to have a terminating black cell,
	//		but I'm hope this will work well enough that I can
	//		use it to design special-purpose puzzles for lectures.
	//
	for (h = 0; h < md->itsGameOf.Crossword2D.itsPuzzleSize; h++)
	{
		for (v = 0; v < md->itsGameOf.Crossword2D.itsPuzzleSize; v++)
		{
			if (md->itsGameOf.Crossword2D.itsSolution[h][v] == '*')
				md->itsGameOf.Crossword2D.itsSolution[h][v] = 'q';
		}
	}
#endif

//	Define a macro SSP to "set scroll position".
#define SSP(h,v)									\
				{									\
					md->itsOffset.itsH = h;			\
					md->itsOffset.itsV = v;			\
					md->itsOffset.itsFlip = false;	\
				}

#ifdef GAME_CONTENT_FOR_SCREENSHOT
	if (IsCurrentLanguage(u"de")) SSP(-1.0/5.0, +1.0/5.0)
	if (IsCurrentLanguage(u"el")) SSP(-2.0/6.0, -1.0/6.0)
	if (IsCurrentLanguage(u"en")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"es")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"fi")) SSP(+3.0/6.0, +3.0/6.0)
	if (IsCurrentLanguage(u"fr")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"it")) SSP(+1.0/5.0, +1.0/5.0)
	if (IsCurrentLanguage(u"ja")) SSP(+2.0/4.0,  0.0/4.0)
	if (IsCurrentLanguage(u"ko")) SSP(-2.0/7.0, -2.0/7.0)
	if (IsCurrentLanguage(u"nl")) SSP(-1.0/5.0, +1.0/5.0)
	if (IsCurrentLanguage(u"pt")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"ru")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"vi")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"zs")) SSP( 0.0/5.0,  0.0/5.0)
	if (IsCurrentLanguage(u"zt")) SSP( 0.0/5.0,  0.0/5.0)
#endif

#ifdef MAKE_GAME_CHOICE_ICONS
	SSP(+1.0/8.0, +1.0/8.0)
#endif
#ifdef MAKE_GAME_CHOICE_ICONS
#warning For thicker lines, go to TorusGamesFragmentFunction2DGrid()	\
	and temporarily set cntHalfWidth to 1.0/32.0
#endif

	//	Set the clue.  (May or may not be redundant -- see comment immediately above.)
	CrosswordRefreshMessage(md);
	
	//	Clear itsPendingCharacter and itsPendingHangulSyllable.
	md->itsGameOf.Crossword2D.itsPendingCharacter		= NO_PENDING_CHARACTER;
	md->itsGameOf.Crossword2D.itsPendingHangulSyllable	= EMPTY_HANGUL_SYLLABLE;

	//	Abort any pending simulation.
	SimulationEnd(md);

	//	The game is not over.
	md->itsGameIsOver	= false;
	md->itsFlashFlag	= false;

CleanUpCrosswordReset:

	GEOMETRY_GAMES_ASSERT(
		theErrorMessage == NULL,
		"Invalid Crossword puzzle data.  See theErrorMessage for details.");
}


static void CrosswordHandMoved(ModelData *md)
{
	SetHotWordFromCursor(md);
}


static void SetHotWordFromCursor(ModelData *md)
{
	double			theRawH,
					theRawV;
	unsigned int	h,
					v;
	double			theOffsetH,
					theOffsetV;
	bool			theFlip;
	WordOrientation	theWordOrientation;

	//	The hand always has coordinates in the range [-0.5, +0.5],
	//	corresponding to cell coordinates h and v in the range [0, n].

	theRawH = md->itsGameOf.Crossword2D.itsPuzzleSize*(md->its2DHandPlacement.itsH + 0.5);
	theRawV = md->itsGameOf.Crossword2D.itsPuzzleSize*(md->its2DHandPlacement.itsV + 0.5);

	h = (unsigned int) floor(theRawH + 0.5);
	v = (unsigned int) floor(theRawV + 0.5);
	
	theOffsetH = theRawH - h;
	theOffsetV = theRawV - v;
	
	//	Note the tentative hand chirality,
	//	which we'll correct below in the special case of v == n in a Klein bottle.
	theFlip = md->its2DHandPlacement.itsFlip;

	//	Wrap the boundary cases  h == n  and/or  v == n .
	//	This code also guards against an "impossible" hand placement,
	//	giving legal (h,v) in all cases.
	if (v >= md->itsGameOf.Crossword2D.itsPuzzleSize)
	{
		v = 0;
		if (md->itsTopology == Topology2DKlein)
		{
			h		= md->itsGameOf.Crossword2D.itsPuzzleSize - h;
			theFlip	= ! theFlip;
		}
	}
	if (h >= md->itsGameOf.Crossword2D.itsPuzzleSize)
		h = 0;
	
	//	Ignore hand cursor motion over black squares.
	if (md->itsGameOf.Crossword2D.itsSolution[h][v] == '*')
		return;
	
	//	Select a horizontal word or a vertical one?
	theWordOrientation = (
		(md->itsGameOf.Crossword2D.itsLeftwardWord[h][v] ?
			(theOffsetV - theOffsetH >= 0.0) : (theOffsetV + theOffsetH >= 0.0))
		? WordHorizontal : WordVertical);
	
	//	Let the hand cursor pass its chirality to the cell it lies on.
	md->itsGameOf.Crossword2D.itsBoardFlips[h][v] = theFlip;
	
	//	Set the word the passes through the selected cell in the selected direction.
	SetHotWord(md, h, v, theFlip, theWordOrientation);
}

static void SetHotWord(
	ModelData		*md,
	unsigned int	h,
	unsigned int	v,
	bool			aFlip,
	WordOrientation	aWordOrientation)
{
	Char16			theDirection;	//	← → ↓
	unsigned int	hh,
					vv;

	//	Which direction does the word run?
	if (aWordOrientation == WordHorizontal)
		theDirection = (md->itsGameOf.Crossword2D.itsLeftwardWord[h][v] ? u'←' : u'→');
	else
		theDirection = u'↓';

	//	If the hot cell hasn't changed, we're done.
	if (md->itsGameOf.Crossword2D.itsHotCellH		== h
	 && md->itsGameOf.Crossword2D.itsHotCellV		== v
	 && md->itsGameOf.Crossword2D.itsHotCellFlip	== aFlip
	 && md->itsGameOf.Crossword2D.itsHotDirection	== theDirection)
	{
		return;
	}

	//	Set the new hot cell.
	md->itsGameOf.Crossword2D.itsHotCellH		= h;
	md->itsGameOf.Crossword2D.itsHotCellV		= v;
	md->itsGameOf.Crossword2D.itsHotCellFlip	= aFlip;
	md->itsGameOf.Crossword2D.itsHotDirection	= theDirection;

	//	Clear any previous hot word.
	for (hh = md->itsGameOf.Crossword2D.itsPuzzleSize; hh-- > 0; )
		for (vv = md->itsGameOf.Crossword2D.itsPuzzleSize; vv-- > 0; )
			md->itsGameOf.Crossword2D.itsHotWord[hh][vv] = false;

	//	Mark the hot word's letters.
	hh = h;
	vv = v;
	do
	{
		md->itsGameOf.Crossword2D.itsHotWord[hh][vv] = true;
		MoveToNextNonblackCell(md, &hh, &vv, NULL, theDirection, true);
	} while (hh != h || vv != v);

	//	Set the clue.
	CrosswordRefreshMessage(md);
}

static bool MoveToPrecedingNonblackCell(
	ModelData		*md,
	unsigned int	*h,
	unsigned int	*v,
	bool			*aFlip,			//	may be NULL
	Char16			aDirection,		//	forward direction of word at (*h, *v)
	bool			aCyclicFlag)	//	wrap around cyclically?
{
	switch (aDirection)
	{
		case u'←':	return MoveToNextNonblackCell(md, h, v, aFlip, u'→', aCyclicFlag);
		case u'→':	return MoveToNextNonblackCell(md, h, v, aFlip, u'←', aCyclicFlag);
		case u'↓':	return MoveToNextNonblackCell(md, h, v, aFlip, u'↑', aCyclicFlag);
		case u'↑':	return MoveToNextNonblackCell(md, h, v, aFlip, u'↓', aCyclicFlag);
		default:	return false;	//	should never occur
	}
}

static bool MoveToNextNonblackCell(
	ModelData		*md,
	unsigned int	*h,
	unsigned int	*v,
	bool			*aFlip,			//	may be NULL
	Char16			aDirection,
	bool			aCyclicFlag)	//	wrap around cyclically?
{
	unsigned int	n,
					theNextH,
					theNextV;
	bool			theNextFlip;
	
	if (aCyclicFlag)
	{
		//	Handle the cyclic case in terms of the non-cyclic case.

		if ( ! MoveToNextNonblackCell(md, h, v, aFlip, aDirection, false))
		{
			while (MoveToPrecedingNonblackCell(md, h, v, aFlip, aDirection, false))
				;
		}
		
		return true;	//	cyclic case is never blocked
	}
	else
	{
		//	Handle the non-cyclic case.

		n = md->itsGameOf.Crossword2D.itsPuzzleSize;

		theNextFlip = (aFlip != NULL ? *aFlip : false);

		switch (aDirection)
		{
			case u'←':
				if (*h > 0)
					theNextH = *h - 1;
				else
					theNextH =  n - 1;
				theNextV = *v;
				break;

			case u'→':
				if (*h < n - 1)
					theNextH = *h + 1;
				else
					theNextH = 0;
				theNextV = *v;
				break;

			case u'↓':
				theNextH = *h;
				if (*v > 0)
					theNextV = *v - 1;
				else
				{
					theNextV =  n - 1;
					if (md->itsTopology == Topology2DKlein)
					{
						theNextH = (theNextH > 0) ? n - theNextH : 0;
						theNextFlip = ! theNextFlip;
					}
				}
				break;

			case u'↑':
				theNextH = *h;
				if (*v < n - 1)
					theNextV = *v + 1;
				else
				{
					theNextV =  0;
					if (md->itsTopology == Topology2DKlein)
					{
						theNextH = (theNextH > 0) ? n - theNextH : 0;
						theNextFlip = ! theNextFlip;
					}
				}
				break;
				
			default:
				return false;	//	should never occur
		}
		
		if (md->itsGameOf.Crossword2D.itsSolution[theNextH][theNextV] != '*')
		{
			*h = theNextH;
			*v = theNextV;
			if (aFlip != NULL)
				*aFlip = theNextFlip;
			
			return true;
		}
		else
			return false;
	}
}

void CrosswordRefreshMessage(ModelData *md)	//	display the current clue
{
	Byte			(*theClueIndexArray)[MAX_CROSSWORD_SIZE];	//	pointer to array of dimensions [MAX_CROSSWORD_SIZE][MAX_CROSSWORD_SIZE]
	unsigned int	theClueIndex;
	Char16			*theClue;
	
	static Char16			theEmptyClue[1] = {0};
	static ColorP3Linear	theClueColor = {0.0, 0.0, 0.5, 1.0};

	theClueIndexArray	= (md->itsGameOf.Crossword2D.itsHotDirection == u'↓' ?
							md->itsGameOf.Crossword2D.itsClueIndexDown :
							md->itsGameOf.Crossword2D.itsClueIndexAcross);
	theClueIndex		= theClueIndexArray[md->itsGameOf.Crossword2D.itsHotCellH][md->itsGameOf.Crossword2D.itsHotCellV];

	if (theClueIndex < MAX_CROSSWORD_CLUES)
		theClue = md->itsGameOf.Crossword2D.itsCluesText[theClueIndex];
	else
		theClue = theEmptyClue;

	SetTorusGamesStatusMessage(theClue, theClueColor, Game2DCrossword);
}


static bool CrosswordDragBegin(
	ModelData	*md,
	bool		aRightClick)
{
	UNUSED_PARAMETER(aRightClick);

	//	Set the hot cell...
	SetHotWordFromCursor(md);

	//	...and then treat *all* mouse clicks as scrolls.
	return false;
}


static TitledErrorMessage *CrosswordCharacterInput(	//	returns NULL or theWantsHanziMessage
	ModelData	*md,
	Char16		aCharacter)
{
	unsigned int	h,
					v;
	bool			theFlip;
	unsigned int	hh,
					vv;
	bool			ff;

	//	This static struct will still exist when the caller receives it,
	//	although obviously it isn't thread-safe.
	static TitledErrorMessage	theWantsHanziMessage;

	//	Just to be safe...
	if (md->itsGameOf.Crossword2D.itsPuzzleSize < 1
	 || md->itsGameOf.Crossword2D.itsPuzzleSize > MAX_CROSSWORD_SIZE
	 || md->itsGameOf.Crossword2D.itsHotCellH >= md->itsGameOf.Crossword2D.itsPuzzleSize
	 || md->itsGameOf.Crossword2D.itsHotCellV >= md->itsGameOf.Crossword2D.itsPuzzleSize)
		return NULL;
	
	//	Let arrows ← ↑ → ↓ move the hot cell.
	if (RespondToArrow(md, aCharacter))
		return NULL;

	//	Ignore character input once the game is solved.
	if (md->itsGameIsOver)
		return NULL;
	
	//	Let a tab '\t' move the hot cell forwards.
	if (aCharacter == u'\t')
	{
		MoveToNextNonblackCell(	md,
								&md->itsGameOf.Crossword2D.itsHotCellH,
								&md->itsGameOf.Crossword2D.itsHotCellV,
								&md->itsGameOf.Crossword2D.itsHotCellFlip,
								md->itsGameOf.Crossword2D.itsHotDirection,
								true);	//	wrap cyclically
		return NULL;
	}
	
	//	If the hot cell is non-empty, 
	//		then "backspace" will delete its contents.
	//	If the hot cell is empty, then let "backspace" delete
	//		the contents of the previous cell instead.
	if (aCharacter == u'\b')
	{
		//	For legibility
		h = md->itsGameOf.Crossword2D.itsHotCellH;
		v = md->itsGameOf.Crossword2D.itsHotCellV;
		
		//	Is hot cell empty?
		if (md->itsGameOf.Crossword2D.itsBoard[h][v] == u' ')
		{
			//	Let a backspace '\b' move the hot cell backwards.
			//	In a moment we'll delete its contents.
			MoveToPrecedingNonblackCell(md,
										&md->itsGameOf.Crossword2D.itsHotCellH,
										&md->itsGameOf.Crossword2D.itsHotCellV,
										&md->itsGameOf.Crossword2D.itsHotCellFlip,
										md->itsGameOf.Crossword2D.itsHotDirection,
										true);	//	wrap cyclically
		}
	}

	//	For easy legibility
	h		= md->itsGameOf.Crossword2D.itsHotCellH;
	v		= md->itsGameOf.Crossword2D.itsHotCellV;
	theFlip	= md->itsGameOf.Crossword2D.itsHotCellFlip;

	//	If the user typed a backspace '\b', 
	//	we might have already moved the hot cell backwards (see above).
	//	Now delete the hot cell's contents and return.
	if (aCharacter == u'\b')
	{
		aCharacter = u' ';

		md->itsGameOf.Crossword2D.itsBoard[h][v]		= aCharacter;
		md->itsGameOf.Crossword2D.itsBoardFlips[h][v]	= theFlip;

		md->itsGameOf.Crossword2D.itsPendingCharacter		= NO_PENDING_CHARACTER;
		md->itsGameOf.Crossword2D.itsPendingHangulSyllable	= EMPTY_HANGUL_SYLLABLE;

		return NULL;
	}

	//	In a Korean puzzle, re-interpret Latin characters by mapping 
	//	the QWERTY keyboard layout to the Dubeolshik (두벌식) layout.
	//
	//		Minor advantage:  Mac users can solve the Korean puzzles
	//			without needing to switch to (or even enable) 
	//			a Korean keyboard layout.
	//
	//		Major advantage:  Windows users, by choosing the Latin keyboard layout,
	//			can assemble jamo into syllables directly within
	//			the crossword puzzle itself (just like on the Mac)
	//			rather than in a separate Hangul composition window.
	//			Either way, the user types as if s/he were using
	//			the Dubeolshik (두벌식) layout.
	//
	//	This operation is case sensitve (for example, 'p' → 'ㅔ' while 'P' → 'ㅖ')
	//	so call QWERTYtoDubeolshik() before calling ToLowerCase().
	if (IsCurrentLanguage(u"ko"))
		aCharacter = QWERTYtoDubeolshik(aCharacter);
	
	//	In a Chinese puzzle, the user might type the comma
	//	in the middle of a proverb like 冰冻三尺，非一日之寒,
	//	if only out of habit.  Ignore such commas.
	if (IsCurrentLanguage(u"zs")
	 || IsCurrentLanguage(u"zt"))
	{
		if (aCharacter == 0xFF0C	//	Chinese regular comma '，'
		 || aCharacter == 0x3001	//	Chinese enumeration comma '、'
		 || aCharacter == u','	)	//	Western comma ','
		{
			return NULL;
		}
	}

	//	In a Chinese puzzle, if the user mistakenly enters
	//	Latin letters thinking the word should be written in pinyin,
	//	display a message asking him/her to enter hanzi.
	if (IsCurrentLanguage(u"zs")
	 || IsCurrentLanguage(u"zt"))
	{
		if (aCharacter < 0x4E00 /* '一' */
		 || aCharacter > 0x9FA5 /* '龥' */)
		{
			theWantsHanziMessage.itsMessage = GetLocalizedText(u"CrosswordWantsHanzi-Message"	);
			theWantsHanziMessage.itsTitle	= GetLocalizedText(u"CrosswordWantsHanzi-Title"		);

			return &theWantsHanziMessage;
		}
	}

	//	Convert all input to lowercase (in languages which make
	//	an uppercase/lowercase distinction -- for all other languages
	//	this call should have no effect).
	aCharacter = ToLowerCase(aCharacter);

	//	In the unlikely (but nevertheless possible) event
	//	that a Japanese user types a half-width katakana,
	//	promote it to a full-width katakana, for example ﾐ → ミ.
	aCharacter = PromoteHalfWidthKatakana(aCharacter);
	
	//	Promote small kana to large kana, for example っ → つ.
	aCharacter = PromoteSmallKana(aCharacter);

	//	Remove Greek tonos (stress-accent) and convert final 'ς' to 'σ'.
	aCharacter = RemoveTonos(aCharacter);
	aCharacter = ConvertFinalSigma(aCharacter);
	
	//	In a Dutch puzzle, merge a newly arrived 'j' with an existing 'i'
	//	to get the combined character 'ĳ'.
	if (IsCurrentLanguage(u"nl"))
	{
		if (md->itsGameOf.Crossword2D.itsBoard[h][v] == u'i'
		 && aCharacter == u'j')
		{
			aCharacter = u'ĳ';
		}
	}
	
	//	In a Japanese puzzle, manually compose Latin letters into kana,
	//	as an alternative to the Japanese IME.
	//	This will let people with no Japanese IME use the puzzles,
	//	and may also be more convenient even for Japanese users,
	//	because they won't have to keep hitting return after each letter.
	//
	//	In a Korean puzzle, manually compose jamo into syllables.
	//	This will require the user to use a Hangul keyboard layout.
	//
	//	In a Russian puzzle, transliterate Latin letters
	//	to Cyrillic ('b'→'б', 'zh'→'ж', etc.) as an alternative
	//	to the Russian IME.  This will let people with no Russian IME
	//	use the puzzles.  Most Russians, however, will prefer
	//	direct Cyrillic input.
	//
	//	In a Greek puzzle, transliterate Latin letters to Greek 
	//	('α'→'a', 'β'→'b', etc.) as an alternative to the Greek keyboard.
	//	The Greek keyboard layout closely follows the American layout.
	//
	//	For Latin-based puzzles, provide some minimal accent support,
	//	but don't pretend to provide full support;  Vietnamese users
	//	may either enter unaccented letters (as is customary in Vietnamese
	//	crosswords, as well as French and Italian ones) and wait 
	//	for the accents to appear at the end, or they may use a Vietnamese IME
	//	in the Windows version if one is available.
	//
	//	In all cases, if aCharacter contributes to itsPendingCharacter
	//	but does not generate a character for immediate use,
	//	the Compose<Script>() function returns a space ' '.
	//	The space ' ' then gets written to the current hot cell,
	//	and serves as a neutral background for displaying itsPendingCharacter.
	//
	//	The Win32 Korean Input Method Editor (IME) passes along
	//	the return or newline used to terminate the final syllable.
	//	To avoid overwriting the next crossword cell, 
	//	ComposeHangul() returns 0 whenever a return or newline
	//	arrives without an internally pending character for it to resolve.
	if (IsCurrentLanguage(u"ja"))
		aCharacter = ComposeHiragana(aCharacter, &md->itsGameOf.Crossword2D.itsPendingCharacter);
	else
	if (IsCurrentLanguage(u"ko"))
		aCharacter = ComposeHangul(aCharacter, &md->itsGameOf.Crossword2D.itsPendingCharacter, &md->itsGameOf.Crossword2D.itsPendingHangulSyllable);
	else
	if (IsCurrentLanguage(u"ru"))
		aCharacter = ComposeCyrillic(md->itsGameOf.Crossword2D.itsBoard[h][v], aCharacter);
	else
	if (IsCurrentLanguage(u"el"))
		aCharacter = ComposeGreek(aCharacter);
	else
		aCharacter = ComposeLatin(aCharacter, &md->itsGameOf.Crossword2D.itsPendingCharacter);
	
	//	ComposeHangul() returns 0 when a spurious return or newline arrives.
	if (aCharacter == 0)
		return NULL;
	
	//	Auto-advance feature
	//
	//	Users often make the following mistake.  Say the user has already
	//	written the horizontal word "dad", and then clicks on the first 'd'
	//	to find the clue "Man's best friend" for the vertical word.
	//	Seeing "d__", the user naturally types 'o' and 'g', and is
	//	surprised and disappointed to see the result "og_".
	//	The problem is that the 'o' overwrote the 'd' that was already present,
	//	then the 'g' followed where the 'o' should have gone.
	//	The technically correct solution would be either to overwrite the initial 'd',
	//	typing 'd', 'o', 'g', or to click into the middle square and type 'o', 'g'
	//	from there.  Nevertheless, given how common this error is, it seems worth
	//	making an exception to the usual rules to automatically correct it.
	//	More precisely, we make the following auto-advance rule:
	//		
	//		If the current square already contains the correct letter
	//		and the user types a different one (other than a space ' '),
	//		then the hot cell advances to that first empty square and 
	//		the letter gets inserted there.
	//
	//	In general I prefer to avoid ad hoc rules, but in the present case
	//	it seems justified.
	if
	(
		md->itsGameOf.Crossword2D.itsBoard[h][v] == md->itsGameOf.Crossword2D.itsSolution[h][v]
	 &&
		aCharacter != md->itsGameOf.Crossword2D.itsSolution[h][v]
	 &&
		(	
			//	A "true" space is a space with no pending character.
			//	A "true" space should never trigger auto-advance,
			//	but should always erase the cell, even if the existing
			//	character there is correct.
			aCharacter != u' '
		 || md->itsGameOf.Crossword2D.itsPendingCharacter != NO_PENDING_CHARACTER
		)
	)
	{
		hh = h;
		vv = v;
		ff = theFlip;
		
		while (md->itsGameOf.Crossword2D.itsBoard[hh][vv] != ' '
			&& MoveToNextNonblackCell(	md,
										&hh,
										&vv,
										&ff,
										md->itsGameOf.Crossword2D.itsHotDirection,
										false))
			;
		
		if (md->itsGameOf.Crossword2D.itsBoard[hh][vv] == ' ')
		{
			md->itsGameOf.Crossword2D.itsHotCellH		= hh;
			md->itsGameOf.Crossword2D.itsHotCellV		= vv;
			md->itsGameOf.Crossword2D.itsHotCellFlip	= ff;

			h		= hh;
			v		= vv;
			theFlip	= ff;
		}
	}

	//	Record aCharacter in the current hot cell.
	md->itsGameOf.Crossword2D.itsBoard[h][v]		= aCharacter;
	md->itsGameOf.Crossword2D.itsBoardFlips[h][v]	= theFlip;
	
	//	If a complete character arrived (not just a contribution to the pending character),
	//	then advance the hot cell.
	//	Exception:  In a Dutch puzzle, if the user typed an 'i' in a cell 
	//	whose solution character is 'ĳ', leave the hot cell there
	//	so the user may easily type a 'j' on top of the 'i'.
	if (aCharacter != ' '
	 && (	md->itsGameOf.Crossword2D.itsSolution[h][v] != u'ĳ'
		 || aCharacter != u'i'))
	{
		MoveToNextNonblackCell(	md,
								&md->itsGameOf.Crossword2D.itsHotCellH,
								&md->itsGameOf.Crossword2D.itsHotCellV,
								&md->itsGameOf.Crossword2D.itsHotCellFlip,
								md->itsGameOf.Crossword2D.itsHotDirection,
								true);
	}

	//	Is that a win?
	if (CrosswordIsSolved(md))
	{
		md->itsGameIsOver = true;
		EnqueueSoundRequest(u"CrosswordComplete.mid");
		FixAccentsAsNecessary(md);
		SimulationBegin(md, Simulation2DCrosswordFlash);
	}
	
	return NULL;
}


static bool RespondToArrow(
	ModelData	*md,
	Char16		aCharacter)
{
	switch (aCharacter)
	{
		case u'←':
		case u'↑':
			//	Move to the preceding cell.
			MoveToPrecedingNonblackCell(md,
										&md->itsGameOf.Crossword2D.itsHotCellH,
										&md->itsGameOf.Crossword2D.itsHotCellV,
										&md->itsGameOf.Crossword2D.itsHotCellFlip,
										md->itsGameOf.Crossword2D.itsHotDirection,
										true);	//	wrap cyclically
			return true;

		case u'→':
		case u'↓':
			//	Move to the next cell.
			MoveToNextNonblackCell(	md,
									&md->itsGameOf.Crossword2D.itsHotCellH,
									&md->itsGameOf.Crossword2D.itsHotCellV,
									&md->itsGameOf.Crossword2D.itsHotCellFlip,
									md->itsGameOf.Crossword2D.itsHotDirection,
									true);	//	wrap cyclically
			return true;
		
		default:
			return false;
	}
}


static bool CrosswordIsSolved(ModelData *md)
{
	unsigned int	h,
					v;

	for (h = 0; h < md->itsGameOf.Crossword2D.itsPuzzleSize; h++)
		for (v = 0; v < md->itsGameOf.Crossword2D.itsPuzzleSize; v++)
			if (md->itsGameOf.Crossword2D.itsSolution[h][v] != '*'
			 && BaseLetter(md->itsGameOf.Crossword2D.itsBoard[h][v]) != BaseLetter(md->itsGameOf.Crossword2D.itsSolution[h][v]))
				return false;

	return true;
}


static void FixAccentsAsNecessary(ModelData *md)
{
	unsigned int	h,
					v;

	for (h = 0; h < md->itsGameOf.Crossword2D.itsPuzzleSize; h++)
	{
		for (v = 0; v < md->itsGameOf.Crossword2D.itsPuzzleSize; v++)
		{
			if (md->itsGameOf.Crossword2D.itsSolution[h][v] != '*'
			 && (	md->itsGameOf.Crossword2D.itsSolution[h][v] < u'ぁ'	//	don't replace board's katakana
				 || md->itsGameOf.Crossword2D.itsSolution[h][v] > u'ヾ')	//		with solution's hiragana
			 && BaseLetter(md->itsGameOf.Crossword2D.itsBoard[h][v]) == BaseLetter(md->itsGameOf.Crossword2D.itsSolution[h][v]))
			{
				md->itsGameOf.Crossword2D.itsBoard[h][v] = md->itsGameOf.Crossword2D.itsSolution[h][v];
			}
		}
	}
}


static void CrosswordSimulationUpdate(ModelData *md)
{
	switch (md->itsSimulationStatus)
	{
		case Simulation2DCrosswordFlash:

			//	Color the background green for three seconds.
			if (md->itsSimulationElapsedTime < 3.0)
			{
				md->itsFlashFlag = true;
			}
			else
			{
				SimulationEnd(md);
				md->itsFlashFlag = false;
			}

			break;

		default:
			break;
	}
}


unsigned int GetNum2DCrosswordBackgroundTextureRepetitions(void)
{
	return NUM_2D_BACKGROUND_TEXTURE_REPETITIONS_CROSSWORD;
}

void Get2DCrosswordKleinAxisColors(
	float	someKleinAxisColors[2][4])	//	output;  premultiplied alpha
{
	someKleinAxisColors[0][0] = 1.00;
	someKleinAxisColors[0][1] = 0.00;
	someKleinAxisColors[0][2] = 0.00;
	someKleinAxisColors[0][3] = 1.00;

	someKleinAxisColors[1][0] = 0.00;
	someKleinAxisColors[1][1] = 0.00;
	someKleinAxisColors[1][2] = 1.00;
	someKleinAxisColors[1][3] = 1.00;
}


unsigned int GetNum2DCrosswordSprites(
	unsigned int	aPuzzleSize)
{
	return aPuzzleSize * aPuzzleSize	//	cells
			+ 1							//	word-direction arrow
			+ 1;						//	background (for flashing)
}

void Get2DCrosswordSpritePlacements(
	ModelData		*md,				//	input
	unsigned int	aNumSprites,		//	input
	Placement2D		*aPlacementBuffer)	//	output;  big enough to hold aNumSprites Placement2D's
{
	unsigned int	n;				//	crossword size is n cells by n cells
	double			theCellSize;	//	1/n
	unsigned int	h,
					v,
					i;

	GEOMETRY_GAMES_ASSERT(
		md->itsGame == Game2DCrossword,
		"Game2DCrossword must be active");

	n			= md->itsGameOf.Crossword2D.itsPuzzleSize;
	theCellSize	= 1.0 / (double) n;

	GEOMETRY_GAMES_ASSERT(
		aNumSprites == GetNum2DCrosswordSprites(n),
		"Internal error:  wrong buffer size");

	//	cells
	//		note (h,v) indexing -- not (row,col)
	for (h = 0; h < n; h++)
	{
		for (v = 0; v < n; v++)
		{
			i = n*h + v;
			aPlacementBuffer[i].itsH		= -0.5 + h * theCellSize;
			aPlacementBuffer[i].itsV		= -0.5 + v * theCellSize;
			aPlacementBuffer[i].itsFlip		= md->itsGameOf.Crossword2D.itsBoardFlips[h][v];
			aPlacementBuffer[i].itsAngle	= 0.0;
			aPlacementBuffer[i].itsSizeH	= theCellSize;
			aPlacementBuffer[i].itsSizeV	= theCellSize;
		}
	}
	
	//	word-direction arrow
	
	//	Start with a default placement that's safe
	//	even when the game is over and no hot cell is defined.
	aPlacementBuffer[n*n].itsH		= 0.0;
	aPlacementBuffer[n*n].itsV		= 0.0;
	aPlacementBuffer[n*n].itsFlip	= false;
	aPlacementBuffer[n*n].itsAngle	= 0.0;
	aPlacementBuffer[n*n].itsSizeH	= 0.25 * theCellSize;
	aPlacementBuffer[n*n].itsSizeV	= 0.25 * theCellSize;

	//	If the game isn't yet over,
	//	fill in the word-direction arrow placement
	//	according to whether the hot word
	//	runs downward, leftward or rightward.
	if ( ! md->itsGameIsOver )
	{
		h = md->itsGameOf.Crossword2D.itsHotCellH;
		v = md->itsGameOf.Crossword2D.itsHotCellV;

		aPlacementBuffer[n*n].itsH = -0.5 + h * theCellSize;
		aPlacementBuffer[n*n].itsV = -0.5 + v * theCellSize;

		if (md->itsGameOf.Crossword2D.itsHotDirection == u'↓')
		{
			//	vertical hot word
			aPlacementBuffer[n*n].itsV		+= -0.25 * theCellSize;
			aPlacementBuffer[n*n].itsAngle	= -0.5 * PI;
		}
		else
		{
			//	horizontal hot word (leftward or rightward)
			aPlacementBuffer[n*n].itsH		+= (md->itsGameOf.Crossword2D.itsLeftwardWord[h][v] ?
												-0.25 * theCellSize :
												+0.25 * theCellSize );
			aPlacementBuffer[n*n].itsFlip	= md->itsGameOf.Crossword2D.itsLeftwardWord[h][v];
		}
	}
	
	//	background (for flashing)
	aPlacementBuffer[n*n + 1] = (Placement2D) {0.0, 0.0, false, 0.0, 1.0, 1.0};
}
